
/*= DIGISTANT 6706 ========================================================*/
/* LabWindows Instrument Driver                                            */
/* Original Release: V1.0  13.10.94                                        */
/* By: M.Westermann                                                        */
/* Originally written in C                                                 */
/* Modification History: By T. Meder    V 1.1   13.05.2002                 */
/*	   Delay added in function "bp6706_read_msg_gpib"					   */
/*=========================================================================*/
#include <rs232.h>
#include <formatio.h>
#include <utility.h>
#include <gpib.h>
#include <ansi_c.h>
#include "bp6706.h"


/*= STATIC VARIABLES ======================================================*/
static int interface;                     /* 0 = GPIB; 1 = Serial*/
static int setup_first ;                  /* remark the first initial*/
static double std_timeout;                /* remark the user timeout*/
static int max_bcc_repeat;                /* Max-Bcc-Repeat */
static int bcc_repeat;                    /* Bcc-Repeat counter*/
static int bcc_state;                     /* Blockcheck state ON or OFF*/
static int instr_cnt;
static int   bp6706_err;

static int address[bp6706_MAX_INSTR + 1];  /*GPIB-Adress*/
static int bd[bp6706_MAX_INSTR + 1];       /*Gpib Board*/
static char cmd[bp6706_MAX_CMD+1];
static char buf[bp6706_MAX_CMD+1];
static char out_buf[bp6706_MAX_CMD+1];       /* buffer for serial send */
static char in_buf[bp6706_MAX_CMD+1];        /* buffer for serial receive */
static char swap_buf[bp6706_MAX_CMD+1];      /* global serial buffer */
static int port_addr[4] ;                    /* RS232-Port Array*/
static int int_level[4] ;                    /* RS232-Interrupt-Level Array*/
static int dev_port[bp6706_MAX_INSTR];       /* Device Port*/
static int dev_group_addr[bp6706_MAX_INSTR]; /* Device Group serial Adress*/
static int dev_user_addr[bp6706_MAX_INSTR];  /* Device User serial Adress */

static char * volt_range[6];
static char * volt_sense[2];
static char * curr_range[4];
static char * volt_unit[3];
static char * curr_unit[2];
static char * system_keyboard[2];
static char * cal_state[2];
static char * cal_curr_low[6];
static char * cal_curr_high[6];
static char * cal_volt_low[10];
static char * cal_volt_high[10];
static char * cal_curr_range[6];
static char * cal_volt_range[10];



/*= UTILITY ROUTINES ======================================================*/

int _VI_FUNC  bp6706_invalid_integer_range (int, int, int, int);
int _VI_FUNC  bp6706_invalid_long_range (long ,long ,long ,int);
int _VI_FUNC  bp6706_invalid_real_range (double, double, double, int);
int _VI_FUNC  bp6706_open_instr_gpib(int );
int _VI_FUNC  bp6706_device_closed_gpib (int);
int _VI_FUNC  bp6706_close_instr_gpib (int);
int _VI_FUNC  bp6706_read_msg_gpib (int,char*,int);
int _VI_FUNC  bp6706_write_msg_gpib (int,char *, int);
void bp6706_setup_arrays (void);
int _VI_FUNC  bp6706_get_status(int);
int _VI_FUNC  bp6706_calculate_bcc(char* );
int _VI_FUNC  bp6706_send_x3_28(int,char*,char*);
int _VI_FUNC  bp6706_send_x3_28_message(int,char*,char*);
int _VI_FUNC  bp6706_receive_x3_28(int,char*,char*,char*);
int _VI_FUNC  bp6706_receive_and_ack(int,char*);
int _VI_FUNC  bp6706_close_instr_serial (int);
int _VI_FUNC  bp6706_device_closed_serial (int);
int _VI_FUNC  bp6706_open_instr_serial (int,long,int,int);


/*=========================================================================*/
/* This function opens the instrument, queries the instrument for its ID,  */
/* and initializes the instrument to a known state.                        */
/*=========================================================================*/
int _VI_FUNC  bp6706_init_gpib(int addr,int reset,char *model,int* instrID)
{
    int ID;
													
    bp6706_err = 0;
    interface = bp6706_GPIB;            /* GPIB */

    if (bp6706_invalid_integer_range (addr, 0, 30,  -1) != 0)
        return bp6706_err;
    if (bp6706_invalid_integer_range (reset, 0, 1,  -2) != 0)
        return bp6706_err;

    ID = bp6706_open_instr_gpib (addr);
    if (ID <= 0)
      return bp6706_err;

    if (reset == 1)
    {
     if ((ibclr (bd[ID]) & (int)0x8000 != 0) != 0)
     {
      bp6706_close_instr_gpib (ID);
      bp6706_err = 224;
      return bp6706_err;
     }

     if (bp6706_send_scpi (ID, "*RST") != 0)
     {
      bp6706_close_instr_gpib (ID);
      return bp6706_err;
     }
    }

    if (bp6706_receive_scpi(ID,"*IDN?",model) != 0)
    {
     bp6706_close_instr_gpib (ID);
     return bp6706_err;
    }
    /*  terminate read on LF  */
    if (ibeos (bd[ID], 0x40a) < 0)
    {
     bp6706_err = 237;
     return bp6706_err;
    }
    *instrID = ID;

    bp6706_setup_arrays ();
    return(0);
}
/*=========================================================================*/
/* This function opens a com port for the instrument module                */
/*=========================================================================*/
int _VI_FUNC  bp6706_init_serial(int port,long baud_rate,double timeout,int set_bcc_state,
                   int g_addr,int u_addr,char* model,int reset,int *instrID)
{
    int ID;
    interface = bp6706_SERIAL;   /* Serial */

    bp6706_err = 0;
    bp6706_setup_arrays ();

    if (bp6706_invalid_integer_range (port, 1,4, -1) != 0)
     return bp6706_err;
    if (bp6706_invalid_long_range (baud_rate, 1200L, 19200L, -2) != 0)
     return bp6706_err;
    if (bp6706_invalid_real_range (timeout, 0.0F, 72000.0F, -3) != 0)
     return bp6706_err;
    if (bp6706_invalid_integer_range (set_bcc_state, 0,1, -4) != 0)
     return bp6706_err;
    if (bp6706_invalid_integer_range (g_addr, 0,15, -5) != 0)
     return bp6706_err;
    if (bp6706_invalid_integer_range (u_addr, 0,15, -6) != 0)
     return bp6706_err;
    if (bp6706_invalid_integer_range (reset, 0,1, -7) != 0)
     return bp6706_err;

    ID = bp6706_open_instr_serial(port,baud_rate,g_addr,u_addr);
    if(ID <= 0)
     return bp6706_err;

    SetComTime (port, timeout);
    std_timeout = timeout;

    if (rs232err != 0)
        return (rs232err);

    max_bcc_repeat = 5;       /* default*/
    bcc_state = set_bcc_state;

    if (reset == 1)
    {
     if (bp6706_send_scpi (ID, "*RST") != 0)
     {
      bp6706_close_instr_serial(ID);
      return bp6706_err;
     }
    }
    if (bp6706_receive_scpi(ID,"*IDN?",model) != 0)
    {
     bp6706_close_instr_serial (ID);
     return bp6706_err;
    }
    *instrID = ID;

    return(0);
}
/* ========================================================================= */
/*  This function puts the device into local mode.                              */
/* ========================================================================= */
int _VI_FUNC  bp6706_local_gpib(int instrID)
{
    bp6706_err = 0;
    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;
    if (bp6706_device_closed_gpib (instrID) != 0)
        return bp6706_err;
    if ((ibloc (bd[instrID]) & (int)0x8000 != 0) != 0)
    {
        bp6706_err = 234;
        return bp6706_err;
    }
    return bp6706_err;
}
/* ========================================================================= */
/*  This function configure the voltage mode                                 */
/* ========================================================================= */
int _VI_FUNC  bp6706_set_volt_config(int instrID,int range,double limit,int sense)
{
    bp6706_err = 0;

    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;
    if (bp6706_invalid_integer_range (range, 0, 5,  -2) != 0)
        return bp6706_err;
    if (bp6706_invalid_integer_range (sense, 0, 1,  -3) != 0)
        return bp6706_err;

    if(range == 5) /* Range 180V */
    {
     if (bp6706_invalid_real_range (limit, 0.1F, 20.0F,  -4) != 0)
       return bp6706_err;

     Fmt (cmd, "%s<SOUR:CURR:PROT2 %fMA",limit);
    }
    else
    {
     if (bp6706_invalid_real_range (limit, 0.1F, 200.0F,  -4) != 0)
      return bp6706_err;

     Fmt (cmd, "%s<SOUR:CURR:PROT1 %fMA",limit);
    }

    if (bp6706_send_scpi (instrID, cmd) != 0)
     return bp6706_err;

    if (bp6706_send_scpi (instrID,volt_range[range]) != 0)
     return bp6706_err;

    if (bp6706_send_scpi (instrID, volt_sense[sense]) != 0)
     return bp6706_err;

    return bp6706_err;
}

/* ========================================================================= */
/*  This function return the configure of the voltage mode                                 */
/* ========================================================================= */
int _VI_FUNC  bp6706_get_volt_config(int instrID,int *range,double *limit,int *sense)
{
    int flag;
    double value;
    char help[5];

    bp6706_err = 0;
    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;

    /*----------- It's the device in AUTO-Mode-----------------*/

    if(bp6706_receive_scpi(instrID,"SOUR:VOLT:RANG:AUTO?",buf) != 0)
     return bp6706_err;

    if (Scan (buf, "%s>%d", &flag) != 1)
    {
      bp6706_err = 236;
      return bp6706_err;
    }
    if(flag == 1)   /* Auto Range */
     *range = 0;
    else            /* Range*/
    {
     if(bp6706_receive_scpi(instrID,"SOUR:VOLT:RANG?",buf) != 0)
      return bp6706_err;

     if (Scan (buf, "%s>%f", &value) != 1)
     {
      bp6706_err = 236;
      return bp6706_err;
     }
     if(value <= 0.02F) /* 20mV */
      *range = 1;
     else if(value <= 0.2F) /* 200mV */
      *range = 2;
     else if(value <= 2.0F) /* 2V */
      *range = 3;
     else if(value <= 20.0F) /* 20V */
      *range = 4;
     else                    /* 180V*/
      *range = 5;
    }

    if(*range == 5)
     Fmt (cmd, "%s<SOUR:CURR:PROT2?\n");
    else
     Fmt (cmd, "%s<SOUR:CURR:PROT1?\n");

    if(bp6706_receive_scpi(instrID,cmd,buf) != 0)
     return bp6706_err;

    if (Scan (buf, "%s>%f", &value) != 1)
    {
     bp6706_err = 236;
     return bp6706_err;
    }
    *limit = value * 1000.0F;   /* in mA */

    if(bp6706_receive_scpi(instrID,"SOUR:VOLT:ALC:SOUR?",buf) != 0)
     return bp6706_err;


    CopyString (help, 0, buf, 0, 3);
    if(CompareStrings(help,0,"INT",0,1) == 0)
     *sense = 0;
    else if(CompareStrings(help,0,"EXT",0,1) == 0)
     *sense = 1;
    else
     *sense = -1;

    return bp6706_err;
}
/* ========================================================================= */
/*  This function configure the current mode                                 */
/* ========================================================================= */
int _VI_FUNC  bp6706_set_curr_config(int instrID,int range,double limit)
{
    bp6706_err = 0;

    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;
    if (bp6706_invalid_integer_range (range, 0, 3,  -2) != 0)
        return bp6706_err;

    if(range == 3) /* Range 200mA */
    {
     if (bp6706_invalid_real_range (limit, 0.1F, 20.0F,  -3) != 0)
       return bp6706_err;

     Fmt (cmd, "%s<SOUR:VOLT:PROT2 %fV",limit);
    }
    else
    {
     if (bp6706_invalid_real_range (limit, 0.1F, 180.0F,  -3) != 0)
      return bp6706_err;

     Fmt (cmd, "%s<SOUR:VOLT:PROT1 %fV",limit);
    }

    if (bp6706_send_scpi (instrID, cmd) != 0)
     return bp6706_err;

    if (bp6706_send_scpi (instrID,curr_range[range]) != 0)
     return bp6706_err;

    return bp6706_err;
}

/* ========================================================================= */
/*  This function return the configure of the voltage mode                                 */
/* ========================================================================= */
int _VI_FUNC  bp6706_get_curr_config(int instrID,int *range,double *limit)
{
    int flag;
    double value;

    bp6706_err = 0;
    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;

    /*----------- It's the device in AUTO-Mode-----------------*/

    if(bp6706_receive_scpi(instrID,"SOUR:CURR:RANG:AUTO?",buf) != 0)
     return bp6706_err;

    if (Scan (buf, "%s>%d", &flag) != 1)
    {
      bp6706_err = 236;
      return bp6706_err;
    }
    if(flag == 1)   /* Auto Range */
     *range = 0;
    else            /* Range*/
    {
     if(bp6706_receive_scpi(instrID,"SOUR:CURR:RANG?",buf) != 0)
      return bp6706_err;

     if (Scan (buf, "%s>%f", &value) != 1)
     {
      bp6706_err = 236;
      return bp6706_err;
     }
     if(value <= 0.002F)     /* 2mA */
      *range = 1;
     else if(value <= 0.02F) /* 20mA */
      *range = 2;
     else                    /* 200mA */
      *range = 3;
    }

    if(*range == 3)
     Fmt (cmd, "%s<SOUR:VOLT:PROT2?\n");
    else
     Fmt (cmd, "%s<SOUR:VOLT:PROT1?\n");

    if(bp6706_receive_scpi(instrID,cmd,buf) != 0)
     return bp6706_err;

    if (Scan (buf, "%s>%f", &value) != 1)
    {
     bp6706_err = 236;
     return bp6706_err;
    }
    *limit = value ;   /* in V */

    return bp6706_err;
}
/* ========================================================================= */
/*  This function output the voltage value                                   */
/* ========================================================================= */
int _VI_FUNC  bp6706_set_volt_value(int instrID,double value,int unit)
{
    bp6706_err = 0;

    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;

    if (bp6706_invalid_integer_range (unit, 0, 2,  -2) != 0)
        return bp6706_err;

    Fmt (cmd, "%s<SOUR:VOLT %f%s",value,volt_unit[unit]);
    if(interface == bp6706_SERIAL)
    {
     if (bp6706_send_scpi (instrID, cmd) != 0)
     {
      if(bp6706_err == 300)  /* NAK by serial */
      {
       bp6706_get_status(instrID);
       return bp6706_err;
      }
      else
       return bp6706_err;
     }
    }
    else     /* GPIB */
    {
     if (bp6706_send_scpi (instrID, cmd) != 0)
      return bp6706_err;

     if(bp6706_get_status(instrID) != 0)
      return bp6706_err;
    }
    return(0);
}
/* ========================================================================= */
/*  This function return output the voltage value                            */
/* ========================================================================= */
int _VI_FUNC  bp6706_get_volt_value(int instrID,double *value)
{
    bp6706_err = 0;

    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;


    Fmt (cmd, "%s<SOUR:VOLT?");
    if(bp6706_receive_scpi(instrID,cmd,buf) != 0)
     return bp6706_err;

    if (Scan (buf, "%s>%f", value) != 1)
    {
     bp6706_err = 236;
     return bp6706_err;
    }
    return(0);
}

/* ========================================================================= */
/*  This function output the current value                                   */
/* ========================================================================= */
int _VI_FUNC  bp6706_set_curr_value(int instrID,double value,int unit)
{
    bp6706_err = 0;

    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;

    if (bp6706_invalid_integer_range (unit, 0, 1,  -2) != 0)
        return bp6706_err;

    Fmt (cmd, "%s<SOUR:CURR %f%s",value,curr_unit[unit]);
    if(interface == bp6706_SERIAL)
    {
     if (bp6706_send_scpi (instrID, cmd) != 0)
     {
      if(bp6706_err == 300)  /* NAK by serial */
      {
       bp6706_get_status(instrID);
       return bp6706_err;
      }
      else
       return bp6706_err;
     }
    }
    else     /* GPIB */
    {
     if (bp6706_send_scpi (instrID, cmd) != 0)
      return bp6706_err;

     if(bp6706_get_status(instrID) != 0)
      return bp6706_err;
    }
    return(0);
}
/* ========================================================================= */
/*  This function return output the current value                            */
/* ========================================================================= */
int _VI_FUNC  bp6706_get_curr_value(int instrID,double *value)
{
    bp6706_err = 0;

    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;


    Fmt (cmd, "%s<SOUR:CURR?");
    if(bp6706_receive_scpi(instrID,cmd,buf) != 0)
     return bp6706_err;

    if (Scan (buf, "%s>%f", value) != 1)
    {
     bp6706_err = 236;
     return bp6706_err;
    }
    *value *= 1000.0F;
    return(0);
}
/* ========================================================================= */
/*  This function return the System Error                                    */
/* ========================================================================= */
int _VI_FUNC  bp6706_system_error(int instrID,char* system_error)
{
    bp6706_err = 0;

    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;

    if(bp6706_receive_scpi(instrID,"SYST:ERR?",system_error) != 0)
     return bp6706_err;
    return 0;
}
/* ========================================================================= */
/*  This function return the System Error                                    */
/* ========================================================================= */
int _VI_FUNC  bp6706_system_keyboard(int instrID,int flag)
{
    bp6706_err = 0;

    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;
    if (bp6706_invalid_integer_range (flag, 0, 1,  -2) != 0)
        return bp6706_err;

    if (bp6706_send_scpi (instrID, system_keyboard[flag]) != 0)
     return bp6706_err;

    return 0;
}
/* ========================================================================= */
/*  This function switch the calibration state ON/OFF                        */
/* ========================================================================= */
int _VI_FUNC  bp6706_cal_state(int instrID,int state)
{
    bp6706_err = 0;

    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;
    if (bp6706_invalid_integer_range (state, 0, 1,  -2) != 0)
        return bp6706_err;

    if (bp6706_send_scpi (instrID, cal_state[state]) != 0)
     return bp6706_err;

    return 0;
}
/* ========================================================================= */
/*  This function adjust the calibration current range ;cal_mode = 0 => low       */
/* ========================================================================= */
int _VI_FUNC  bp6706_cal_curr_adjust(int instrID,int range,int cal_mode)
{
    bp6706_err = 0;

    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;
    if (bp6706_invalid_integer_range (range, 0, 5,  -2) != 0)
        return bp6706_err;

    if (bp6706_invalid_integer_range (cal_mode, 0, 1,  -3) != 0)
        return bp6706_err;

    if(cal_mode == 0)    /* LOW*/
    {
     if (bp6706_send_scpi (instrID, cal_curr_low[range]) != 0)
      return bp6706_err;
    }
    else
    {
     if (bp6706_send_scpi (instrID, cal_curr_high[range]) != 0)
      return bp6706_err;
    }

    return 0;
}
/* ========================================================================= */
/*  This function adjust the calibration voltage range                        */
/* ========================================================================= */
int _VI_FUNC  bp6706_cal_volt_adjust(int instrID,int range,int cal_mode)
{
    bp6706_err = 0;

    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;
    if (bp6706_invalid_integer_range (range, 0, 9,  -2) != 0)
        return bp6706_err;
    if (bp6706_invalid_integer_range (cal_mode, 0, 1,  -3) != 0)
        return bp6706_err;

    if(cal_mode ==  0)  /* low*/
    {
     if (bp6706_send_scpi (instrID, cal_volt_low[range]) != 0)
      return bp6706_err;
    }
    else
    {
     if (bp6706_send_scpi (instrID, cal_volt_high[range]) != 0)
      return bp6706_err;
    }

    return 0;
}
/* ========================================================================= */
/*  This function adjust the calibration protection current range             */
/* ========================================================================= */
int _VI_FUNC  bp6706_cal_protect_curr_adjust(int instrID,int cal_mode)
{
    bp6706_err = 0;

    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;
    if (bp6706_invalid_integer_range (cal_mode, 0, 1,  -2) != 0)
        return bp6706_err;

    if(cal_mode == 0)   /* LOW*/
    {
     if (bp6706_send_scpi (instrID, "CAL:PROT:CURR:LOW") != 0)
      return bp6706_err;
    }
    else
    {
     if (bp6706_send_scpi (instrID, "CAL:PROT:CURR:HIGH") != 0)
      return bp6706_err;
    }
    return 0;
}
/* ========================================================================= */
/*  This function adjust the calibration protection voltage range                   */
/* ========================================================================= */
int _VI_FUNC  bp6706_cal_protect_volt_adjust(int instrID,int cal_mode)
{
    bp6706_err = 0;

    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;
    if (bp6706_invalid_integer_range (cal_mode, 0, 1,  -2) != 0)
        return bp6706_err;

    if(cal_mode == 0)   /* LOW*/
    {
     if (bp6706_send_scpi (instrID, "CAL:PROT:VOLT:LOW") != 0)
      return bp6706_err;
    }
    else
    {
     if (bp6706_send_scpi (instrID, "CAL:PROT:VOLT:HIGH") != 0)
      return bp6706_err;
    }

    return 0;
}
/* ========================================================================= */
/*  This function set the calibration current data                          */
/* ========================================================================= */
int _VI_FUNC  bp6706_cal_curr_data(int instrID,int range,double low_data,double high_data)
{
    bp6706_err = 0;

    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;
    if (bp6706_invalid_integer_range (range, 0, 5,  -2) != 0)
        return bp6706_err;

    Fmt (cmd, "%s<CAL:CURR:DATA %s,%f[p10]MA,%f[p10]MA",cal_curr_range[range],
                        low_data*1000.0F,
                        high_data*1000.0F);

    if (bp6706_send_scpi (instrID, cmd) != 0)
     return bp6706_err;

    return 0;
}


/* ========================================================================= */
/*  This function set the calibration voltage data                          */
/* ========================================================================= */
int _VI_FUNC  bp6706_cal_volt_data(int instrID,int range,double low_data,double high_data)
{
    bp6706_err = 0;

    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;
    if (bp6706_invalid_integer_range (range, 0, 9,  -2) != 0)
        return bp6706_err;

    if(range <= 3)     /* mV*/
     Fmt (cmd, "%s<CAL:VOLT:DATA %s,%f[p10]MV,%f[p10]MV",cal_volt_range[range],
                        low_data*1000.0F,
                        high_data*1000.0F);
    else               /* V */
     Fmt (cmd, "%s<CAL:VOLT:DATA %s,%f[p10]V,%f[p10]V",cal_volt_range[range],
                        low_data,
                        high_data);

    if (bp6706_send_scpi (instrID, cmd) != 0)
     return bp6706_err;

    return 0;
}
/* ========================================================================= */
/*  This function set the calibration protect current data                   */
/* ========================================================================= */
int _VI_FUNC  bp6706_cal_protect_curr_data(int instrID,double low_data,double high_data)
{
    bp6706_err = 0;

    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;

    Fmt (cmd, "%s<CAL:PROT:CURR:DATA %f[p10]MA,%f[p10]MA",low_data*1000.0F,high_data*1000.0F);

    if (bp6706_send_scpi (instrID, cmd) != 0)
     return bp6706_err;

    return 0;
}

/* ========================================================================= */
/*  This function set the calibration protect voltage data                   */
/* ========================================================================= */
int _VI_FUNC  bp6706_cal_protect_volt_data(int instrID,double low_data,double high_data)
{
    bp6706_err = 0;

    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
        return bp6706_err;

    Fmt (cmd, "%s<CAL:PROT:VOLT:DATA %f[p10]V,%f[p10]V",low_data,high_data);

    if (bp6706_send_scpi (instrID, cmd) != 0)
     return bp6706_err;

    return 0;
}
/*=========================================================================*/
/* This function check the device dependent error flag                     */
/*=========================================================================*/
int _VI_FUNC  bp6706_get_status(int instrID)
{
      int status_flag,quest_flag;

      bp6706_err = 0;


      if(bp6706_receive_scpi(instrID,"*ESR?",buf) != 0)
       return bp6706_err;

/*      FmtOut("\n Standard Event :%s ",buf);*/

      Scan(buf,"%s>%i",&status_flag);
      if(status_flag & bp6706_DEVICE_DEPT_ERROR)
      {
       if(status_flag & bp6706_EXECUTION_ERROR)
       {
        bp6706_err = 312;    /* Execution ERROR => (Overrange)*/
        return bp6706_err;
       }

       if(bp6706_receive_scpi(instrID,"STAT:QUES:COND?",buf) != 0)
        return bp6706_err;

       /*FmtOut("\n Quest Stat :%s ",buf);       */

       Scan(buf,"%s>%i",&quest_flag);

       if(quest_flag == 0)
       {
        bp6706_err = 0;    /* NO ERROR */
        return bp6706_err;
       }
       if(quest_flag & bp6706_VOLTAGE_ERROR)
       {
        bp6706_err = 310;    /* Voltage ERROR */
        return bp6706_err;
       }
       if(quest_flag & bp6706_CURRENT_ERROR)
       {
        bp6706_err = 311;    /* Current ERROR */
        return bp6706_err;
       }
      }

      return bp6706_err;
}
/*=========================================================================*/
/* This function SendSCPI                                                  */
/*=========================================================================*/
int _VI_FUNC  bp6706_send_scpi(int instrID,char* send_buffer)
{
   bp6706_err = 0;

   Fmt(cmd,"%s<%s\n",send_buffer);

   if(interface == bp6706_GPIB)  /* GPIB ?? */
   {
    bp6706_err = bp6706_write_msg_gpib(instrID,cmd,NumFmtdBytes ());

    if(bp6706_err != 0)
     return (bp6706_err);
   }
   else    /* Serial */
   {
    bp6706_err = bp6706_send_x3_28(instrID,"sr",cmd);
    if(bp6706_err != 0)
     return (bp6706_err);
   }

   return(0);
}
/*=========================================================================*/
/* This function receive SCPI                                              */
/*=========================================================================*/
int _VI_FUNC  bp6706_receive_scpi(int instrID,char* send_buffer,char* receive_buffer)
{
   bp6706_err = 0;

   Fmt(cmd,"%s<%s\n",send_buffer);

  /* FmtOut("\n++%s++",cmd);*/

   if(interface == bp6706_GPIB)  /* GPIB ?? */
   {
    bp6706_err = bp6706_write_msg_gpib(instrID,cmd,NumFmtdBytes ());
    if(bp6706_err != 0)
     return (bp6706_err);

    bp6706_err = bp6706_read_msg_gpib(instrID,receive_buffer,bp6706_MAX_CMD);
    if(bp6706_err != 0)
     return (bp6706_err);
   }
   else
   {
    bp6706_err = bp6706_send_x3_28(instrID,"sr",cmd);
    if(bp6706_err != 0)
     return (bp6706_err);

    bp6706_err = bp6706_receive_x3_28(instrID,"po","",receive_buffer);
    if(bp6706_err != 0)
     return (bp6706_err);
   }
  /* FmtOut("\n**%s**",receive_buffer);*/

   return(0);
}
/* ========================================================================= */
/*  This function closes the instrument.                                     */
/* ========================================================================= */
int _VI_FUNC  bp6706_close (int instrID)
{
    bp6706_err = 0;

    if (bp6706_invalid_integer_range (instrID, 1, bp6706_MAX_INSTR,  -1) != 0)
     return (bp6706_err);

    if(interface == bp6706_GPIB)  /* GPIB ?? */
    {
     if (bp6706_device_closed_gpib (instrID) != 0)
      return (bp6706_err);
     bp6706_close_instr_gpib (instrID);
    }
    else
    {
     if (bp6706_device_closed_serial (instrID) != 0)
      return (bp6706_err);
     bp6706_close_instr_serial (instrID);
    }
    return (bp6706_err);
}
/* = GPIB UTILITY ROUTINES == ============================================== */
/* ========================================================================= */
/*  This function locates and initializes an entry in the Instrument Table   */
/*  and the GPIB device table for the instrument.  If successful, the        */
/*  instrument ID is returned, else a -1 is returned.  The size of the       */
/*  Instrument Table can be changed in the include file by altering the      */
/*  constant bp6706_MAX_INSTR.                                               */
/* ========================================================================= */
int _VI_FUNC  bp6706_open_instr_gpib(int addr)
{
    int i;
    int instrID;

    instrID = 0;
    /*  Check to see if the instrument is already in the Instrument Table.  */
    for (i = 1; i <= bp6706_MAX_INSTR; i++)
        if (address[i] == addr)
        {
            instrID = i;
            i = bp6706_MAX_INSTR;
        }
    /*  If it is not in the instrument table, open an entry for the instrument.  */
    if (instrID <= 0)
        for (i = 1; i <= bp6706_MAX_INSTR; i++)
            if (!address[i])
            {
                instrID = i;
                i = bp6706_MAX_INSTR;
            }
    /*  If an entry could not be opened in the Instrument Table, return an error. */
    if (instrID <= 0)
    {
        bp6706_err = 220;
        return  -1;
    }
    /*   If the device has not been opened in the GPIB device table (bd[ID] = 0), */
    /*   then open it.                                                            */
    if (bd[instrID] <= 0)
    {
        if (instr_cnt <= 0)
            CloseInstrDevs ("bp6706");
        bd[instrID] = OpenDev ("", "bp6706");
        if (bd[instrID] <= 0)
        {
            bp6706_err = 220;
            return  -1;
        }
    }
    /*   Change the primary address of the device     */
    if (ibpad (bd[instrID], addr) < 0)
    {
        bp6706_err = 233;
        return  -1;
    }
    address[instrID] = addr;
    instr_cnt++;
    return instrID;
}

/* ========================================================================== */
/*  This function closes the instrument by removing it from the GPIB device   */
/*  table and setting the address and the bd to zero in the Instrument Table. */
/*  The return value is equal to the global error variable.                   */
/* ========================================================================== */
int _VI_FUNC  bp6706_close_instr_gpib (int instrID)
{

    if (bd[instrID] != 0)  {
        CloseDev (bd[instrID]);
        bd[instrID] = 0;
        address[instrID] = 0;
        instr_cnt--;
    }
    else
        bp6706_err = 221;
    return bp6706_err;
}
/* ========================================================================= */
/*  This function checks to see if the module has been initialized.  If the  */
/*  device has not been opened, a 1 is returned, 0 otherwise.                */
/* ========================================================================= */
int _VI_FUNC  bp6706_device_closed_gpib (int instrID)
{

    if (bd[instrID] <= 0)
    {
        bp6706_err = 232;
        return  -1;
    }
    return 0;
}

/* ========================================================================= */
/*  This function reads a buffer of data from the instrument. The return     */
/*  value is equal to the global error variable.                             */
/* ========================================================================= */
int _VI_FUNC  bp6706_read_msg_gpib(int instrID,char *msg,int cnt)
{
	// Add a Delay, because a timeout has occured
	Delay (0.05);	// 13.05.2002 MT
	
    if (ibrd (bd[instrID], msg, (long)cnt) <= 0)
        bp6706_err = 231;
    else
        bp6706_err = 0;

    /*FmtOut("\n%s",msg);*/

    return bp6706_err;
}

/* ========================================================================= */
/*  This function writes a buffer of data to the instrument. The return      */
/*  value is equal to the global error variable.                             */
/* ========================================================================= */
int _VI_FUNC  bp6706_write_msg_gpib (int instrID,char *msg,int cnt)
{
/*    FmtOut("\n%s",msg);*/
    if (ibwrt (bd[instrID], msg, (long)cnt) <= 0)
        bp6706_err = 230;
    else
        bp6706_err = 0;
    return bp6706_err;
}
/* = SERIAL UTILITY ROUTINES ============================================== */
/* ========================================================================= */
/*  This function locates and initializes an entry in the Instrument Table   */
/*  and the device table for the instrument.  If successful, the            */
/*  instrument ID is returned, else a -1 is returned.  The size of the       */
/*  Instrument Table can be changed in the include file by altering the      */
/*  constant bp6706_MAX_INSTR.                                               */
/* ========================================================================= */
int _VI_FUNC  bp6706_open_instr_serial (int port,long baud_rate,int g_addr,int u_addr)
{
    int i, instrID;

    instrID = 0;

    /*--Check to see if the instrument is already in the Instrument Table.-*/

    for (i=1; i< bp6706_MAX_INSTR; i++)
    {
/*    FmtOut("\n g_adr %i u_adr %i i= %i",dev_group_addr[i],dev_user_addr[i],i);*/
     if((dev_group_addr[i] == g_addr) && (dev_user_addr[i] == u_addr))
     {
        instrID = i;
        i = bp6706_MAX_INSTR;
     }
    }

   /*------ If it is not in the instrument table, open an entry for the
            instrument. ---------------------------------------------------*/

    if (instrID <= 0)
    {
     for (i=1; i< bp6706_MAX_INSTR; i++)
     {
/*     FmtOut("\n g_adr %i u_adr %i i= %i",dev_group_addr[i],dev_user_addr[i],i);*/
        if( (dev_group_addr[i] < 0) && (dev_user_addr[i] < 0))
        {
         instrID = i;
         i = bp6706_MAX_INSTR;
        }
     }
    }

    /*----- If an entry could not be opened in the Instrument Table,
            return an error.-----------------------------------------------*/

    if (instrID <= 0)
    {
     bp6706_err = 220;
     return -1;
    }


    dev_port[instrID] = port;
    /*OpenComConfig (1, 9600L, 0, 7, 1, 512, 512, 0, 0);*/
    OpenComConfig (dev_port[instrID],"",baud_rate,0, 8, 1, 512, 512);
    if (rs232err != 0)
    {
     dev_port[instrID] = 0;
     dev_group_addr[instrID] = -1;
     dev_user_addr[instrID] = -1;
     bp6706_err = rs232err;
     return (-1);
    }
    dev_group_addr[instrID] = g_addr;
    dev_user_addr[instrID] = u_addr;
    return instrID;
}
/*=========================================================================*/
/* Function: Close Instrument                                              */
/* Purpose:  This function closes the instrument by removing it from the   */
/*           device table and setting the address and the bd to zero       */
/*           in the Instrument Table.  The return value is equal to the    */
/*           global error variable.                                        */
/*=========================================================================*/
int _VI_FUNC  bp6706_close_instr_serial (int instrID)
{
    if( (dev_group_addr[instrID] >= 0) && (dev_user_addr[instrID] >= 0))
    {
      dev_port[instrID] = 0;
      dev_group_addr[instrID] = -1;
      dev_user_addr[instrID] = -1;
    }
    else
     bp6706_err = 221;

    return bp6706_err;
 }
/*=========================================================================*/
/* Function: Device Closed                                                 */
/* Purpose:  This function checks to see if the module has been            */
/*           initialized.  If the device has not been opened, a 1 is       */
/*           returned, 0 otherwise.                                        */
/*=========================================================================*/
int _VI_FUNC  bp6706_device_closed_serial (int instrID)
{
     if( (dev_group_addr[instrID] < 0) && (dev_user_addr[instrID] < 0))
     {
        bp6706_err = 232;
        return -1;
     }
     return 0;
}

/*=========================================================================*/
/* This function send a comand and message and wait for ACK                */
/*=========================================================================*/
int _VI_FUNC  bp6706_send_x3_28(int instrID,char* cmd,char* msg)
{
    int ret,r_byte;

    FlushInQ(dev_port[instrID]);           /* Clear Port */

    bcc_repeat = 0;
    for(;;)
    {
     ret = bp6706_send_x3_28_message(instrID,cmd,msg);
     if(ret != 0)
       return (ret);     /* ERROR*/

     r_byte = ComRdByte(dev_port[instrID]);
     if (rs232err != 0)
      return (231);

    /* FmtOut("\nsend ret %c %d\n",r_byte,bcc_repeat);*/
     Delay(0.01);
     switch(r_byte)
     {
      case ACK:
       return  (0);
       break;
      case NAK:
       if(bcc_repeat >= max_bcc_repeat)
        return (300);                /* NAK => MAX Repeat */
       bcc_repeat++;
       break;
      case EOT:
       return (301);
       break;
     }
    }

    return (0);
}

/*=========================================================================*/
/* This function send comand and message and receive a string              */
/*=========================================================================*/
int _VI_FUNC  bp6706_receive_x3_28(int instrID,char* cmd,char* msg,char* receive_buf)
{
    int ret;

    FlushInQ(dev_port[instrID]);           /* Clear Port */
    ret = bp6706_send_x3_28_message(instrID,cmd,msg);
    if(ret != 0)
      return (ret);     /* ERROR*/

    ret = bp6706_receive_and_ack(instrID,receive_buf);
    if(ret != 0)
      return (ret);     /* ERROR*/

    ComRdByte(dev_port[instrID]);   /* Read the EOT from Port*/

    return(0);
}
/*=========================================================================*/
/* This function receive a string                                          */
/*=========================================================================*/
int _VI_FUNC  bp6706_receive_and_ack(int instrID,char* receive_buf)
{
    int r_byte,r_bcc,bcc,l,first_byte;

    bcc_repeat = 0;
    for(;;)
    {
     first_byte = ComRdByte(dev_port[instrID]);   /* Read the EOT or STX from Port*/
     if(first_byte == EOT)
      return (301);

     r_byte = ComRdTerm(dev_port[instrID],in_buf,bp6706_MAX_CMD-1,ETX);/* Read to ETX*/
     if (rs232err != 0)
      return (231);

     if(bcc_state == 1)
     {
      r_bcc = ComRdByte(dev_port[instrID]);   /* Read the BCC from Port*/
      if (rs232err != 0)
       return (231);
     }
     in_buf[r_byte] = ETX;            /* Calculate bcc with ETX */
     in_buf[r_byte+1] = NUL;          /* Terminate the String*/
     l = StringLength(in_buf);

    /* FmtOut("\nin %c %s",first_byte,in_buf);*/

     CopyString (swap_buf, 0,in_buf, 0, l);   /* calculate bcc without STX */
     if(bcc_state == 1)
     {
      bcc = bp6706_calculate_bcc(swap_buf);
      if(r_bcc != bcc)
      {
       ComWrtByte(dev_port[instrID],NAK);
       if(bcc_repeat >= max_bcc_repeat)
        return (300);                /* NAK => MAX Repeat */
       bcc_repeat++;
      }
      else
      {
       ComWrtByte(dev_port[instrID],ACK);
       CopyString (receive_buf, 0,in_buf,0,l-3); /* Copy without LF,ETX*/
       return(0);                /* 0 = ACK == OK*/
      }
     }
     else    /* Without bcc check*/
     {
      ComWrtByte(dev_port[instrID],ACK);
      CopyString (receive_buf, 0,in_buf,0,l-3); /* Copy without LF,ETX*/
      return(0);                /* 0 = ACK == OK*/
     }
    }

    return(0);
}

/*=========================================================================*/
/* This function send a comand and message                                 */
/*=========================================================================*/
int _VI_FUNC  bp6706_send_x3_28_message(int instrID,char* cmd,char* msg)
{
    int ret,l;
    int bcc;

    ret = CompareStrings (msg, 0,"", 0, 0); /* Send a command without msg ?*/
    if(ret == 0)
     Fmt(out_buf,"%s<%c%x%x%x%x%s%c", EOT,
                                      dev_group_addr[instrID],
                                      dev_group_addr[instrID],
                                      dev_user_addr[instrID],
                                      dev_user_addr[instrID],
                                      cmd,ENQ); /* without msg*/
    else
    {
     Fmt(swap_buf ,"%s<%s%c",msg,ETX);   /* with msg*/
     if(bcc_state == 1)
     {
      bcc = bp6706_calculate_bcc(swap_buf);
      Fmt(out_buf,"%s<%c%x%x%x%x%s%c%s%c", EOT,
                                           dev_group_addr[instrID],
                                           dev_group_addr[instrID],
                                           dev_user_addr[instrID],
                                           dev_user_addr[instrID],
                                           cmd,STX,swap_buf,bcc);
     }
     else
      Fmt(out_buf,"%s<%c%x%x%x%x%s%c%s", EOT,
                                         dev_group_addr[instrID],
                                         dev_group_addr[instrID],
                                         dev_user_addr[instrID],
                                         dev_user_addr[instrID],
                                         cmd,STX,swap_buf);
    }

   /* FmtOut("out %s",out_buf);*/

    l = StringLength(out_buf);
    ret = ComWrt(dev_port[instrID], out_buf, l);     /* Write the Message */
    if (rs232err != 0)
     return (230);

    return (0);
}
/*=========================================================================*/
/* This function calculate the block check                                 */
/*=========================================================================*/
int _VI_FUNC  bp6706_calculate_bcc(char* msg)
{
    int l,i,bcc;

    bcc = 0;
    l = StringLength(msg);
    for(i = 0;i <= l;i++)
     bcc ^= msg[i];     /* exclusive OR */

    bcc |= 0x80;    /* OR Bit 8 */

    return (bcc);
}

/* = Global UTILITY ROUTINES ==============================================*/
/*=========================================================================*/
/* Function: Invalid Integer Range                                         */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int _VI_FUNC  bp6706_invalid_integer_range (int val,int min,int max,int err_code)
{
  if ((val < min) || (val > max))
  {
    bp6706_err = err_code;
    return -1;
  }
  return 0;
}
/*=========================================================================*/
/* Function: Invalid Long Range                                            */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int _VI_FUNC  bp6706_invalid_long_range (long val,long min,long max,int err_code)
{
  if ((val < min) || (val > max))
  {
    bp6706_err = err_code;
    return -1;
  }
  return 0;
}

/*=========================================================================*/
/* Function: Invalid Real Range                                            */
/* Purpose:  This function checks a real number to see if it lies between  */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int _VI_FUNC  bp6706_invalid_real_range (double val,double min,double max,int err_code)
{
  if ((val < min) || (val > max))
  {
    bp6706_err = err_code;
    return -1;
  }
  return 0;
}

void bp6706_setup_arrays (void)
{
    int i;

    if(!setup_first)/*----------- Reset the Arrays before initial --------*/
    {
     for (i=1; i< bp6706_MAX_INSTR; i++)
     {
       dev_group_addr[i] = -1;
       dev_user_addr[i]  = -1;
       dev_port[i]       =  0;
     }
     setup_first = 1;
    }
    port_addr[0] = 0x3f8;
    port_addr[1] = 0x2f8;
    port_addr[2] = 0x3e8;
    port_addr[3] = 0x2e8;

    int_level[0] = 4;
    int_level[1] = 3;
    int_level[2] = 4;
    int_level[3] = 3;

    /*  voltage range  */
    volt_range[0] = "SOUR:VOLT:RANG:AUTO ON";
    volt_range[1] = "SOUR:VOLT:RANG 10MV";
    volt_range[2] = "SOUR:VOLT:RANG 100MV";
    volt_range[3] = "SOUR:VOLT:RANG 1V";
    volt_range[4] = "SOUR:VOLT:RANG 10V";
    volt_range[5] = "SOUR:VOLT:RANG 100V";

    /*  voltage sense  */
    volt_sense[0] = "SOUR:VOLT:ALC:SOUR INT";
    volt_sense[1] = "SOUR:VOLT:ALC:SOUR EXT";

    /*  current range  */
    curr_range[0] = "SOUR:CURR:RANG:AUTO ON";
    curr_range[1] = "SOUR:CURR:RANG 1MA";
    curr_range[2] = "SOUR:CURR:RANG 10MA";
    curr_range[3] = "SOUR:CURR:RANG 100MA";

    /*  voltage unit  */
    volt_unit[0] = "UV";
    volt_unit[1] = "MV";
    volt_unit[2] = "V";

    /*  current unit  */
    curr_unit[0] = "UA";
    curr_unit[1] = "MA";

    /*  SYSTEM Keyboard  */
    system_keyboard[0] = "SYST:KLOC OFF";
    system_keyboard[1] = "SYST:KLOC ON";

    /*  Calibration   */
    cal_state[0] = "CAL:STAT OFF";
    cal_state[1] = "CAL:STAT ON";

    cal_curr_low[0] = "CAL:CURR:LOW:RANG 1MA";
    cal_curr_low[1] = "CAL:CURR:LOW:RANG -1MA";
    cal_curr_low[2] = "CAL:CURR:LOW:RANG 10MA";
    cal_curr_low[3] = "CAL:CURR:LOW:RANG -10MA";
    cal_curr_low[4] = "CAL:CURR:LOW:RANG 100MA";
    cal_curr_low[5] = "CAL:CURR:LOW:RANG -100MA";

    cal_curr_high[0] = "CAL:CURR:HIGH:RANG 1MA";
    cal_curr_high[1] = "CAL:CURR:HIGH:RANG -1MA";
    cal_curr_high[2] = "CAL:CURR:HIGH:RANG 10MA";
    cal_curr_high[3] = "CAL:CURR:HIGH:RANG -10MA";
    cal_curr_high[4] = "CAL:CURR:HIGH:RANG 100MA";
    cal_curr_high[5] = "CAL:CURR:HIGH:RANG -100MA";

    cal_volt_low[0] = "CAL:VOLT:LOW:RANG 10MV";
    cal_volt_low[1] = "CAL:VOLT:LOW:RANG -10MV";
    cal_volt_low[2] = "CAL:VOLT:LOW:RANG 100MV";
    cal_volt_low[3] = "CAL:VOLT:LOW:RANG -100MV";
    cal_volt_low[4] = "CAL:VOLT:LOW:RANG 1V";
    cal_volt_low[5] = "CAL:VOLT:LOW:RANG -1V";
    cal_volt_low[6] = "CAL:VOLT:LOW:RANG 10V";
    cal_volt_low[7] = "CAL:VOLT:LOW:RANG -10V";
    cal_volt_low[8] = "CAL:VOLT:LOW:RANG 100V";
    cal_volt_low[9] = "CAL:VOLT:LOW:RANG -100V";

    cal_volt_high[0] = "CAL:VOLT:HIGH:RANG 10MV";
    cal_volt_high[1] = "CAL:VOLT:HIGH:RANG -10MV";
    cal_volt_high[2] = "CAL:VOLT:HIGH:RANG 100MV";
    cal_volt_high[3] = "CAL:VOLT:HIGH:RANG -100MV";
    cal_volt_high[4] = "CAL:VOLT:HIGH:RANG 1V";
    cal_volt_high[5] = "CAL:VOLT:HIGH:RANG -1V";
    cal_volt_high[6] = "CAL:VOLT:HIGH:RANG 10V";
    cal_volt_high[7] = "CAL:VOLT:HIGH:RANG -10V";
    cal_volt_high[8] = "CAL:VOLT:HIGH:RANG 100V";
    cal_volt_high[9] = "CAL:VOLT:HIGH:RANG -100V";

    cal_curr_range[0] = "1MA";
    cal_curr_range[1] = "-1MA";
    cal_curr_range[2] = "10MA";
    cal_curr_range[3] = "-10MA";
    cal_curr_range[4] = "100MA";
    cal_curr_range[5] = "-100MA";

    cal_volt_range[0] = "10MV";
    cal_volt_range[1] = "-10MV";
    cal_volt_range[2] = "100MV";
    cal_volt_range[3] = "-100MV";
    cal_volt_range[4] = "1V";
    cal_volt_range[5] = "-1V";
    cal_volt_range[6] = "10V";
    cal_volt_range[7] = "-10V";
    cal_volt_range[8] = "100V";
    cal_volt_range[9] = "-100V";

}
/*void main(void)
{
 int instID;
 int range;
 double limit;
 char buf[100];
 int sense;
 bp6706_init_serial (1, 9600, 1.0, 0, 0, 0, buf, 1,&instID );


} */
